<?php

namespace App\Http\Controllers;

use App\Models\Payment;
use App\Models\Student;
use App\Models\Invoice;
use App\Models\GradeFee;
use App\Models\Treasury;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class PaymentController extends Controller
{
    public function create(Request $request)
    {
        $students = Student::all();
        $currencies = [
            'USD' => 'الدولار الأمريكي ($)',
            'EGP' => 'الجنيه المصري (EGP)',
            'SAR' => 'الريال السعودي (SAR)',
            'SDG' => 'الجنيه السوداني (SDG)',
        ];
        $selectedStudent = $request->query('student_id') ?? null;

        return view('payments.create', compact('students', 'currencies', 'selectedStudent'));
    }

public function index(Request $request)
{
    $students = Student::all();

    $currencies = [
        'USD' => 'الدولار الأمريكي ($)',
        'EGP' => 'الجنيه المصري (EGP)',
        'SAR' => 'الريال السعودي (SAR)',
        'SDG' => 'الجنيه السوداني (SDG)',
    ];

    $invoices = Invoice::with(['student', 'payments'])
        ->latest()
        ->paginate(8); // <-- NEW: paginate results

    $stats = [
        'total_invoices' => Invoice::count(),
        'total_amount'   => Invoice::sum('total_amount'),
        'total_paid'     => Invoice::sum('paid_amount'),
        'total_remaining'=> Invoice::sum('remaining_amount'),
    ];

    return view('payments.index', compact('invoices', 'stats', 'students', 'currencies'));
}


    public function getStudentFees($studentId)
    {
        try {
            $student = Student::findOrFail($studentId);
            $fees = GradeFee::where('grade', $student->grade)->get();

            $tuitionFees = $fees->where('type', 'tuition')->sum('amount');
            $regFees = $fees->where('type', 'registration')->sum('amount');
            $totalAmount = $tuitionFees + $regFees;

            $invoice = Invoice::firstOrCreate(
                ['student_id' => $student->id],
                [
                    'total_amount' => $totalAmount,
                    'paid_amount' => 0,
                    'remaining_amount' => $totalAmount,
                    'status' => 'pending',
                    'discount_percent' => 0,
                    'discount_amount' => 0,
                ]
            );

            $payments = Payment::where('invoice_id', $invoice->id)
                ->orderBy('created_at', 'desc')->get();

            return response()->json([
                'success' => true,
                'student' => $student,
                'invoice' => $invoice,
                'payments' => $payments,
                'fees' => $fees
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'حدث خطأ أثناء تحميل بيانات الطالب',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'student_id' => 'required|exists:students,id',
            'invoice_id' => 'required|exists:invoices,id',
            'payment_amount' => 'required|numeric|min:0.01',
            'currency' => 'required|in:USD,EGP,SAR,SDG',
            'exchange_rate' => 'required|numeric|min:0.0001',
            'notes' => 'nullable|string|max:500',
            'payment_date' => 'nullable|date'
        ]);

        if ($validator->fails()) {
            return redirect()
                ->route('payments.index', ['student_id' => $request->student_id])
                ->withErrors($validator)
                ->withInput();
        }

        $validated = $validator->validated();

        try {
            $invoice = Invoice::findOrFail($validated['invoice_id']);
            $student = Student::findOrFail($validated['student_id']);

            // Convert to USD
            $amountUsd = $validated['currency'] === 'USD'
                ? $validated['payment_amount']
                : $validated['payment_amount'] / $validated['exchange_rate'];

            if ($amountUsd > $invoice->remaining_amount) {
                return redirect()
                    ->route('payments.create', ['student_id' => $student->id])
                    ->withErrors([
                        'payment_amount' => 'مبلغ الدفع يتجاوز الرصيد المتبقي في الفاتورة!'
                    ])
                    ->withInput();
            }

            $payment = Payment::create([
                'invoice_id' => $invoice->id,
                'student_id' => $student->id,
                'amount' => $validated['payment_amount'],
                'currency' => $validated['currency'],
                'exchange_rate' => $validated['exchange_rate'],
                'amount_usd' => $amountUsd,
                'notes' => $validated['notes'],
                'payment_date' => $validated['payment_date'] ?? now(),
                'method' => 'cash', // Default method
                'status' => 'completed'
            ]);

            $this->updateTreasury($validated['currency'], $validated['payment_amount']);

            $newPaidAmount = $invoice->paid_amount + $amountUsd;
            $newRemainingAmount = max($invoice->total_amount - $newPaidAmount, 0);

            $invoice->update([
                'paid_amount' => $newPaidAmount,
                'remaining_amount' => $newRemainingAmount,
                'status' => $newRemainingAmount == 0 ? 'paid' :
                            ($newPaidAmount > 0 ? 'partial' : 'pending')
            ]);

            return redirect()->route('payments.create', ['student_id' => $student->id])
                ->with('success', 'تم تسجيل الدفع بنجاح وتحديث الفاتورة!');
        } catch (\Exception $e) {
            return redirect()
                ->route('payments.create', ['student_id' => $request->student_id])
                ->withErrors(['error' => 'حدث خطأ أثناء تسجيل الدفع: ' . $e->getMessage()])
                ->withInput();
        }
    }

    private function updateTreasury($currency, $balance)
    {
        $treasury = Treasury::firstOrCreate(['currency' => $currency], ['balance' => 0]);
        $treasury->increment('balance', $balance);
    }

    public function show($id)
    {
        $payment = Payment::with(['student', 'invoice'])->findOrFail($id);
        return view('payments.show', compact('payment'));
    }

    public function destroy(Payment $payment)
    {
        $payment->delete();
        return redirect()->route('payments.index')->with('success','تم حذف الدفعية بنجاح');
    }
}