<?php

namespace App\Http\Controllers;

use App\Models\Student;
use Illuminate\Http\Request;
use App\Models\GradeFee; 



class StudentController extends Controller
{
    public function index(Request $request)
{
    $query = Student::query();

    // Filter by grade
    if ($request->filled('class_name')) {
        $query->where('grade', $request->class_name);
    }

    // Search by name or admission number
    if ($request->filled('search')) {
        $search = $request->search;
        $query->where(function($q) use ($search) {
            $q->where('fname', 'like', "%$search%")
              ->orWhere('middle_name', 'like', "%$search%")
              ->orWhere('lname', 'like', "%$search%")
              ->orWhere('admission_no', 'like', "%$search%");
        });
    }

    $students = $query->latest()->paginate(10); // paginate

    return view('students.index', compact('students'));
}




public function create()
{
    // Fetch distinct grades from grade_fees
    $grades = GradeFee::distinct()->orderBy('grade')->pluck('grade');

    return view('students.create', compact('grades'));
}


    public function store(Request $request)
{
    $grades = GradeFee::distinct()->pluck('grade')->toArray(); // get array for validation

    $data = $request->validate([
        'fname' => 'required|string|max:255',
        'middle_name' => 'nullable|string|max:255',
        'lname' => 'required|string|max:255',
        'date_of_birth' => 'nullable|date',
        'birth_city' => 'nullable|string|max:255',
        'nationality' => 'nullable|string|max:255',
        'gender' => 'nullable|in:male,female',
        'address' => 'required|string',
        'blood_type' => 'nullable|string|max:5',
        'telephone' => 'required|string|max:20',
        'parent_full_name' => 'nullable|string|max:255',
        'marital_status' => 'nullable|string|max:50',
        'parent_telephone' => 'nullable|string|max:20',
        'whatsapp' => 'nullable|string|max:20',
        'email' => 'nullable|email|max:255',
        'education' => 'nullable|string|max:100',
        'job' => 'nullable|string|max:100',
        'grade' => 'nullable|string|in:' . implode(',', $grades), // dynamic grades
        'academic_year' => 'nullable|string|max:20',
        'has_allergies' => 'nullable|boolean',
        'allergies_details' => 'nullable|string',
        'has_medical_conditions' => 'nullable|boolean',
        'medical_condition_details' => 'nullable|string',
    ]);

    // Serial number generation
    $yearSuffix = date('y'); 
    $countThisYear = Student::whereYear('created_at', date('Y'))->count() + 1;
    $data['serial_number'] = 'S' . $countThisYear . '-' . $yearSuffix;

    Student::create($data);

    return redirect()->route('students.index')->with('success','تم إضافة الطالب بنجاح');
}


    public function edit(Student $student) { return view('students.edit', compact('student')); }
    public function update(Request $request, Student $student)
    {
        $data = $request->validate([
            'fname'=>'required','lname'=>'required','grade'=>'nullable','academic_year'=>'nullable'
        ]);
        $student->update($data);
        return redirect()->route('students.index')->with('success','تم تعديل الطالب');
    }

    public function destroy(Student $student)
    {
        $student->delete();
        return redirect()->route('students.index')->with('success','تم حذف الطالب');
    }

    
}
