@extends('layouts.app')

@section('content')
<div class="container py-4 d-flex justify-content-center align-items-start" style="min-height: 90vh;">
    <div class="w-100" style="max-width: 900px;">
        <h3 class="fw-bold mb-4 text-center text-primary">📋 نظام الفواتير والمدفوعات</h3>

        <!-- Display Success Message -->
        @if(session('success'))
            <div class="alert alert-success alert-dismissible fade show" role="alert">
                {{ session('success') }}
                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
            </div>
        @endif

        <!-- Display Error Messages -->
        @if($errors->any())
            <div class="alert alert-danger alert-dismissible fade show" role="alert">
                <ul class="mb-0">
                    @foreach($errors->all() as $error)
                        <li>{{ $error }}</li>
                    @endforeach
                </ul>
                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
            </div>
        @endif

        <!-- 🔹 اختيار الطالب -->
        <div class="card shadow-sm mb-4 border-0">
            <div class="card-body bg-light rounded-3">
                <div class="mb-3">
                    <label class="fw-semibold mb-2">👨‍🎓 اختر الطالب</label>
                    <select id="studentSelect" class="form-select form-select-lg shadow-sm">
                        <option value="">اختر الطالب</option>
                        @foreach($students as $student)
                            <option value="{{ $student->id }}" 
                                @if($selectedStudent == $student->id) selected @endif>
                                {{ $student->fname }} {{ $student->lname }} - {{ $student->grade }}
                            </option>
                        @endforeach
                    </select>
                </div>
            </div>
        </div>

        <!-- 🔹 تفاصيل الفاتورة -->
        <section id="invoiceDetails" class="mb-3" style="display:none;">
            <div class="card shadow-sm border-0">
                <div class="card-header bg-gradient-primary text-white fw-bold fs-5">📚 تفاصيل الفاتورة</div>
                <div class="card-body bg-light">
                    <div class="alert alert-info mb-3">
                        <strong>👨‍🎓 الطالب:</strong> <span id="studentName"></span><br>
                        <strong>📊 الصف الدراسي:</strong> <span id="studentGrade"></span>
                    </div>
                    
                    <div class="row text-center fw-semibold mb-3">
                        <div class="col-6 col-md-3 mb-2">
                            <div class="p-2 bg-white shadow-sm rounded-3">
                                <div class="text-muted small">إجمالي الفاتورة</div>
                                <div class="fs-5 text-dark">$<span id="totalAmount">0.00</span></div>
                            </div>
                        </div>
                        <div class="col-6 col-md-3 mb-2">
                            <div class="p-2 bg-white shadow-sm rounded-3">
                                <div class="text-muted small">المبلغ المدفوع</div>
                                <div class="fs-5 text-success">$<span id="paidAmount">0.00</span></div>
                            </div>
                        </div>
                        <div class="col-6 col-md-3 mb-2">
                            <div class="p-2 bg-white shadow-sm rounded-3">
                                <div class="text-muted small">المتبقي</div>
                                <div class="fs-5 text-danger">$<span id="remainingAmount">0.00</span></div>
                            </div>
                        </div>
                        <div class="col-6 col-md-3 mb-2">
                            <div class="p-2 bg-white shadow-sm rounded-3">
                                <div class="text-muted small">حالة الفاتورة</div>
                                <div class="fs-5" id="invoiceStatus">-</div>
                            </div>
                        </div>
                    </div>

                    <!-- تفاصيل الرسوم -->
                    <h6 class="fw-bold text-primary mb-2">📋 بنود الرسوم:</h6>
                    <div id="feeItems" class="mb-2"></div>
                </div>
            </div>
        </section>

        <!-- 🔹 تسجيل الدفع -->
        <section id="paymentSection" class="mb-3" style="display:none;">
            <div class="card border-success shadow-sm rounded-3">
                <div class="card-header bg-success bg-gradient text-white fw-bold">💵 تسجيل دفعة جديدة</div>
                <div class="card-body">
                    <form method="POST" action="{{ route('payments.store') }}" id="paymentForm">
                        @csrf
                        <input type="hidden" name="student_id" id="studentId">
                        <input type="hidden" name="invoice_id" id="invoiceId">

                        <div class="row mb-2 g-2">
                            <div class="col-md-6">
                                <label class="fw-semibold">💰 مبلغ الدفع</label>
                                <input type="number" step="0.01" name="payment_amount" id="paymentAmount" 
                                    class="form-control shadow-sm" required placeholder="أدخل المبلغ"
                                    value="{{ old('payment_amount') }}">
                                <small class="text-muted">الرصيد المتبقي: $<span id="remainingBalance">0.00</span></small>
                                @error('payment_amount')
                                    <div class="text-danger small">{{ $message }}</div>
                                @enderror
                            </div>
                            <div class="col-md-6">
                                <label class="fw-semibold">💬 ملاحظات (اختياري)</label>
                                <input type="text" name="notes" class="form-control shadow-sm" 
                                    placeholder="ملاحظات الدفع" value="{{ old('notes') }}">
                            </div>
                        </div>

                        <div class="row g-2 mb-2">
                            <div class="col-md-4">
                                <label class="fw-semibold">💱 العملة</label>
                                <select name="currency" id="currencySelect" class="form-select shadow-sm" required>
                                    <option value="">اختر العملة</option>
                                    @foreach($currencies as $key => $currency)
                                        <option value="{{ $key }}" 
                                            {{ old('currency') == $key ? 'selected' : '' }}>
                                            {{ $currency }}
                                        </option>
                                    @endforeach
                                </select>
                                @error('currency')
                                    <div class="text-danger small">{{ $message }}</div>
                                @enderror
                            </div>
                            <div class="col-md-4">
                                <label class="fw-semibold">📊 سعر الصرف</label>
                                <input type="number" step="0.0001" name="exchange_rate" id="exchangeRate" 
                                    class="form-control shadow-sm" placeholder="سعر الصرف" 
                                    value="{{ old('exchange_rate', '1.0000') }}" required>
                                @error('exchange_rate')
                                    <div class="text-danger small">{{ $message }}</div>
                                @enderror
                            </div>
                            <div class="col-md-4">
                                <label class="fw-semibold">📅 تاريخ الدفع</label>
                                <input type="date" name="payment_date" class="form-control shadow-sm" 
                                    value="{{ old('payment_date', date('Y-m-d')) }}">
                            </div>
                        </div>

                        <button type="submit" class="btn btn-success w-100 shadow-sm py-2 fw-bold">
                            💾 حفظ الدفع وتحديث الفاتورة
                        </button>
                    </form>
                </div>
            </div>
        </section>

        <!-- 🔹 سجل المدفوعات -->
        <section id="paymentHistory" class="mb-3" style="display:none;">
            <div class="card shadow-sm border-0">
                <div class="card-header bg-info bg-gradient text-white fw-bold">📜 سجل المدفوعات</div>
                <div class="card-body p-0">
                    <div class="table-responsive">
                        <table class="table table-bordered table-striped align-middle m-0">
                            <thead class="table-info">
                                <tr class="text-center">
                                    <th>التاريخ</th>
                                    <th>المبلغ</th>
                                    <th>العملة</th>
                                    <th>سعر الصرف</th>
                                    <th>بالدولار</th>
                                    <th>ملاحظات</th>
                                    <th>الحالة</th>
                                </tr>
                            </thead>
                            <tbody id="paymentHistoryBody"></tbody>
                        </table>
                    </div>
                </div>
            </div>
        </section>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', () => {
    // Pre-select student if passed via URL
    const urlParams = new URLSearchParams(window.location.search);
    const studentId = urlParams.get('student_id');
    if (studentId) {
        const select = document.getElementById('studentSelect');
        select.value = studentId;
        select.dispatchEvent(new Event('change'));
    }

    // Set default exchange rate for USD
    const currencySelect = document.getElementById('currencySelect');
    const exchangeRate = document.getElementById('exchangeRate');
    
    currencySelect.addEventListener('change', function() {
        if (this.value === 'USD') {
            exchangeRate.value = '1.0000';
            exchangeRate.readOnly = true;
        } else {
            exchangeRate.value = '';
            exchangeRate.readOnly = false;
            exchangeRate.focus();
        }
    });

    // Trigger change event if USD is selected from old input
    if (currencySelect.value === 'USD') {
        exchangeRate.value = '1.0000';
        exchangeRate.readOnly = true;
    }

    // Dynamic calculation
    const paymentAmountInput = document.getElementById('paymentAmount');
    paymentAmountInput.addEventListener('input', () => {
        const total = parseFloat(document.getElementById('totalAmount').textContent) || 0;
        const paid = parseFloat(document.getElementById('paidAmount').textContent) || 0;
        const entered = parseFloat(paymentAmountInput.value) || 0;
        const remaining = total - paid - entered;
        
        // Update only the remaining balance display
        document.getElementById('remainingBalance').textContent = Math.max(remaining, 0).toFixed(2);
    });
});

// Student selection handler
document.getElementById('studentSelect').addEventListener('change', async function() {
    const studentId = this.value;
    if (!studentId) {
        hideAllSections();
        return;
    }

    // Show all sections
    ['invoiceDetails', 'paymentSection', 'paymentHistory'].forEach(s => {
        const section = document.getElementById(s);
        if (section) section.style.display = 'block';
    });

    // Show loading indicators
    const feeItems = document.getElementById('feeItems');
    const paymentHistoryBody = document.getElementById('paymentHistoryBody');
    if (feeItems) feeItems.innerHTML = '<div class="text-center py-3"><div class="spinner-border text-primary"></div></div>';
    if (paymentHistoryBody) paymentHistoryBody.innerHTML = '<tr><td colspan="7" class="text-center py-3">جارٍ التحميل...</td></tr>';

    try {
        const response = await fetch(`/payments/get-student-fees/${studentId}`);
        const data = await response.json();
        if (data.success) {
            displayStudentData(data);
        } else {
            throw new Error(data.message || 'فشل تحميل البيانات');
        }
    } catch (error) {
        alert('حدث خطأ أثناء تحميل بيانات الطالب: ' + error.message);
        hideAllSections();
    }
});

function displayStudentData(data) {
    // Set form values
    document.getElementById('studentId').value = data.student.id;
    document.getElementById('studentName').textContent = `${data.student.fname} ${data.student.lname}`;
    document.getElementById('studentGrade').textContent = data.student.grade || '-';
    document.getElementById('invoiceId').value = data.invoice.id;
    
    // Format amounts
    const totalAmount = parseFloat(data.invoice.total_amount || 0).toFixed(2);
    const paidAmount = parseFloat(data.invoice.paid_amount || 0).toFixed(2);
    const remainingAmount = parseFloat(data.invoice.remaining_amount || 0).toFixed(2);
    
    document.getElementById('totalAmount').textContent = totalAmount;
    document.getElementById('paidAmount').textContent = paidAmount;
    document.getElementById('remainingAmount').textContent = remainingAmount;
    document.getElementById('remainingBalance').textContent = remainingAmount;
    
    // Set invoice status
    document.getElementById('invoiceStatus').innerHTML = getStatusBadge(data.invoice.status || 'pending');

    // Display fees
    const feeItems = document.getElementById('feeItems');
    if (data.fees && data.fees.length > 0) {
        feeItems.innerHTML = data.fees.map(fee => `
            <div class="d-flex justify-content-between border-bottom py-1">
                <span>${fee.item_name || 'رسوم غير محددة'}</span>
                <span class="fw-bold">$${parseFloat(fee.amount || 0).toFixed(2)}</span>
            </div>
        `).join('');
    } else {
        feeItems.innerHTML = '<div class="text-center text-muted py-2">لا توجد رسوم</div>';
    }

    // Display payment history
    const tbody = document.getElementById('paymentHistoryBody');
    if (data.payments && data.payments.length > 0) {
        tbody.innerHTML = data.payments.map(p => `
            <tr class="text-center">
                <td>${new Date(p.created_at).toLocaleDateString('ar-EG')}</td>
                <td>${parseFloat(p.amount).toFixed(2)}</td>
                <td>${p.currency}</td>
                <td>${parseFloat(p.exchange_rate).toFixed(4)}</td>
                <td>$${parseFloat(p.amount_usd).toFixed(2)}</td>
                <td>${p.notes || '-'}</td>
                <td>${getStatusBadge('paid')}</td>
            </tr>
        `).join('');
    } else {
        tbody.innerHTML = '<tr><td colspan="7" class="text-center text-muted py-2">لا توجد مدفوعات</td></tr>';
    }
}

function getStatusBadge(status) {
    const classes = { 
        'paid': 'badge bg-success', 
        'pending': 'badge bg-warning', 
        'partial': 'badge bg-info', 
        'overdue': 'badge bg-danger' 
    };
    const texts = { 
        'paid': 'مدفوع', 
        'pending': 'معلق', 
        'partial': 'جزئي', 
        'overdue': 'متأخر' 
    };
    return `<span class="${classes[status] || 'badge bg-secondary'}">${texts[status] || status}</span>`;
}

function hideAllSections() {
    ['invoiceDetails', 'paymentSection', 'paymentHistory'].forEach(s => {
        const section = document.getElementById(s);
        if (section) section.style.display = 'none';
    });
}

// Form submission - REMOVED problematic validation
document.getElementById('paymentForm')?.addEventListener('submit', function(e) {
    const btn = this.querySelector('button[type="submit"]');
    if (btn) {
        btn.disabled = true;
        btn.innerHTML = '<span class="spinner-border spinner-border-sm"></span> جاري المعالجة...';
    }
});
</script>

<style>
.container { min-height: calc(100vh - 80px); }
.badge { font-size: 0.75em; padding: 0.35em 0.65em; }
.card { border-radius: 15px; }
.table th { border-top: none; font-weight: 600; }
.form-control:focus, .form-select:focus { box-shadow: 0 0 0 0.25rem rgba(13, 110, 253, 0.25); }
</style>
@endsection